﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/vpc-lattice/VPCLatticeRequest.h>
#include <aws/vpc-lattice/VPCLattice_EXPORTS.h>
#include <aws/vpc-lattice/model/ListenerProtocol.h>
#include <aws/vpc-lattice/model/RuleAction.h>

#include <utility>

namespace Aws {
namespace VPCLattice {
namespace Model {

/**
 */
class CreateListenerRequest : public VPCLatticeRequest {
 public:
  AWS_VPCLATTICE_API CreateListenerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateListener"; }

  AWS_VPCLATTICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID or ARN of the service.</p>
   */
  inline const Aws::String& GetServiceIdentifier() const { return m_serviceIdentifier; }
  inline bool ServiceIdentifierHasBeenSet() const { return m_serviceIdentifierHasBeenSet; }
  template <typename ServiceIdentifierT = Aws::String>
  void SetServiceIdentifier(ServiceIdentifierT&& value) {
    m_serviceIdentifierHasBeenSet = true;
    m_serviceIdentifier = std::forward<ServiceIdentifierT>(value);
  }
  template <typename ServiceIdentifierT = Aws::String>
  CreateListenerRequest& WithServiceIdentifier(ServiceIdentifierT&& value) {
    SetServiceIdentifier(std::forward<ServiceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the listener. A listener name must be unique within a service.
   * The valid characters are a-z, 0-9, and hyphens (-). You can't use a hyphen as
   * the first or last character, or immediately after another hyphen.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateListenerRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The listener protocol.</p>
   */
  inline ListenerProtocol GetProtocol() const { return m_protocol; }
  inline bool ProtocolHasBeenSet() const { return m_protocolHasBeenSet; }
  inline void SetProtocol(ListenerProtocol value) {
    m_protocolHasBeenSet = true;
    m_protocol = value;
  }
  inline CreateListenerRequest& WithProtocol(ListenerProtocol value) {
    SetProtocol(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The listener port. You can specify a value from 1 to 65535. For HTTP, the
   * default is 80. For HTTPS, the default is 443.</p>
   */
  inline int GetPort() const { return m_port; }
  inline bool PortHasBeenSet() const { return m_portHasBeenSet; }
  inline void SetPort(int value) {
    m_portHasBeenSet = true;
    m_port = value;
  }
  inline CreateListenerRequest& WithPort(int value) {
    SetPort(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action for the default rule. Each listener has a default rule. The
   * default rule is used if no other rules match.</p>
   */
  inline const RuleAction& GetDefaultAction() const { return m_defaultAction; }
  inline bool DefaultActionHasBeenSet() const { return m_defaultActionHasBeenSet; }
  template <typename DefaultActionT = RuleAction>
  void SetDefaultAction(DefaultActionT&& value) {
    m_defaultActionHasBeenSet = true;
    m_defaultAction = std::forward<DefaultActionT>(value);
  }
  template <typename DefaultActionT = RuleAction>
  CreateListenerRequest& WithDefaultAction(DefaultActionT&& value) {
    SetDefaultAction(std::forward<DefaultActionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If you retry a request that completed successfully
   * using the same client token and parameters, the retry succeeds without
   * performing any actions. If the parameters aren't identical, the retry fails.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateListenerRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags for the listener.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateListenerRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateListenerRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_serviceIdentifier;
  bool m_serviceIdentifierHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  ListenerProtocol m_protocol{ListenerProtocol::NOT_SET};
  bool m_protocolHasBeenSet = false;

  int m_port{0};
  bool m_portHasBeenSet = false;

  RuleAction m_defaultAction;
  bool m_defaultActionHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace VPCLattice
}  // namespace Aws
