﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudcontrol/CloudControlApiRequest.h>
#include <aws/cloudcontrol/CloudControlApi_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudControlApi {
namespace Model {

/**
 */
class ListResourcesRequest : public CloudControlApiRequest {
 public:
  AWS_CLOUDCONTROLAPI_API ListResourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListResources"; }

  AWS_CLOUDCONTROLAPI_API Aws::String SerializePayload() const override;

  AWS_CLOUDCONTROLAPI_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the resource type.</p>
   */
  inline const Aws::String& GetTypeName() const { return m_typeName; }
  inline bool TypeNameHasBeenSet() const { return m_typeNameHasBeenSet; }
  template <typename TypeNameT = Aws::String>
  void SetTypeName(TypeNameT&& value) {
    m_typeNameHasBeenSet = true;
    m_typeName = std::forward<TypeNameT>(value);
  }
  template <typename TypeNameT = Aws::String>
  ListResourcesRequest& WithTypeName(TypeNameT&& value) {
    SetTypeName(std::forward<TypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For private resource types, the type version to use in this resource
   * operation. If you do not specify a resource version, CloudFormation uses the
   * default version.</p>
   */
  inline const Aws::String& GetTypeVersionId() const { return m_typeVersionId; }
  inline bool TypeVersionIdHasBeenSet() const { return m_typeVersionIdHasBeenSet; }
  template <typename TypeVersionIdT = Aws::String>
  void SetTypeVersionId(TypeVersionIdT&& value) {
    m_typeVersionIdHasBeenSet = true;
    m_typeVersionId = std::forward<TypeVersionIdT>(value);
  }
  template <typename TypeVersionIdT = Aws::String>
  ListResourcesRequest& WithTypeVersionId(TypeVersionIdT&& value) {
    SetTypeVersionId(std::forward<TypeVersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Identity and Access Management (IAM)
   * role for Cloud Control API to use when performing this resource operation. The
   * role specified must have the permissions required for this operation. The
   * necessary permissions for each event handler are defined in the <code> <a
   * href="https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-handlers">handlers</a>
   * </code> section of the <a
   * href="https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html">resource
   * type definition schema</a>.</p> <p>If you do not specify a role, Cloud Control
   * API uses a temporary session created using your Amazon Web Services user
   * credentials.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations.html#resource-operations-permissions">Specifying
   * credentials</a> in the <i>Amazon Web Services Cloud Control API User
   * Guide</i>.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  ListResourcesRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the previous paginated request didn't return all of the remaining results,
   * the response object's <code>NextToken</code> parameter value is set to a token.
   * To retrieve the next set of results, call this action again and assign that
   * token to the request object's <code>NextToken</code> parameter. If there are no
   * remaining results, the previous response object's <code>NextToken</code>
   * parameter is set to <code>null</code>.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListResourcesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListResourcesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource model to use to select the resources to return.</p>
   */
  inline const Aws::String& GetResourceModel() const { return m_resourceModel; }
  inline bool ResourceModelHasBeenSet() const { return m_resourceModelHasBeenSet; }
  template <typename ResourceModelT = Aws::String>
  void SetResourceModel(ResourceModelT&& value) {
    m_resourceModelHasBeenSet = true;
    m_resourceModel = std::forward<ResourceModelT>(value);
  }
  template <typename ResourceModelT = Aws::String>
  ListResourcesRequest& WithResourceModel(ResourceModelT&& value) {
    SetResourceModel(std::forward<ResourceModelT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_typeName;

  Aws::String m_typeVersionId;

  Aws::String m_roleArn;

  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::String m_resourceModel;
  bool m_typeNameHasBeenSet = false;
  bool m_typeVersionIdHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_resourceModelHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudControlApi
}  // namespace Aws
