/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { listExtensions } from '@google/gemini-cli-core';
import { getErrorMessage } from '../../utils/errors.js';
import { MessageType } from '../types.js';
import { CommandKind, } from './types.js';
import open from 'open';
import process from 'node:process';
async function listAction(context) {
    const historyItem = {
        type: MessageType.EXTENSIONS_LIST,
        extensions: context.services.config
            ? listExtensions(context.services.config)
            : [],
    };
    context.ui.addItem(historyItem, Date.now());
}
function updateAction(context, args) {
    const updateArgs = args.split(' ').filter((value) => value.length > 0);
    const all = updateArgs.length === 1 && updateArgs[0] === '--all';
    const names = all ? null : updateArgs;
    if (!all && names?.length === 0) {
        context.ui.addItem({
            type: MessageType.ERROR,
            text: 'Usage: /extensions update <extension-names>|--all',
        }, Date.now());
        return Promise.resolve();
    }
    let resolveUpdateComplete;
    const updateComplete = new Promise((resolve) => (resolveUpdateComplete = resolve));
    const historyItem = {
        type: MessageType.EXTENSIONS_LIST,
        extensions: context.services.config
            ? listExtensions(context.services.config)
            : [],
    };
    updateComplete.then((updateInfos) => {
        if (updateInfos.length === 0) {
            context.ui.addItem({
                type: MessageType.INFO,
                text: 'No extensions to update.',
            }, Date.now());
        }
        context.ui.addItem(historyItem, Date.now());
        context.ui.setPendingItem(null);
    });
    try {
        context.ui.setPendingItem(historyItem);
        context.ui.dispatchExtensionStateUpdate({
            type: 'SCHEDULE_UPDATE',
            payload: {
                all,
                names,
                onComplete: (updateInfos) => {
                    resolveUpdateComplete(updateInfos);
                },
            },
        });
        if (names?.length) {
            const extensions = listExtensions(context.services.config);
            for (const name of names) {
                const extension = extensions.find((extension) => extension.name === name);
                if (!extension) {
                    context.ui.addItem({
                        type: MessageType.ERROR,
                        text: `Extension ${name} not found.`,
                    }, Date.now());
                    continue;
                }
            }
        }
    }
    catch (error) {
        resolveUpdateComplete([]);
        context.ui.addItem({
            type: MessageType.ERROR,
            text: getErrorMessage(error),
        }, Date.now());
    }
    return updateComplete.then((_) => { });
}
async function exploreAction(context) {
    const extensionsUrl = 'https://geminicli.com/extensions/';
    // Only check for NODE_ENV for explicit test mode, not for unit test framework
    if (process.env['NODE_ENV'] === 'test') {
        context.ui.addItem({
            type: MessageType.INFO,
            text: `Would open extensions page in your browser: ${extensionsUrl} (skipped in test environment)`,
        }, Date.now());
    }
    else if (process.env['SANDBOX'] &&
        process.env['SANDBOX'] !== 'sandbox-exec') {
        context.ui.addItem({
            type: MessageType.INFO,
            text: `View available extensions at ${extensionsUrl}`,
        }, Date.now());
    }
    else {
        context.ui.addItem({
            type: MessageType.INFO,
            text: `Opening extensions page in your browser: ${extensionsUrl}`,
        }, Date.now());
        try {
            await open(extensionsUrl);
        }
        catch (_error) {
            context.ui.addItem({
                type: MessageType.ERROR,
                text: `Failed to open browser. Check out the extensions gallery at ${extensionsUrl}`,
            }, Date.now());
        }
    }
}
const listExtensionsCommand = {
    name: 'list',
    description: 'List active extensions',
    kind: CommandKind.BUILT_IN,
    action: listAction,
};
const updateExtensionsCommand = {
    name: 'update',
    description: 'Update extensions. Usage: update <extension-names>|--all',
    kind: CommandKind.BUILT_IN,
    action: updateAction,
    completion: async (context, partialArg) => {
        const extensions = context.services.config
            ? listExtensions(context.services.config)
            : [];
        const extensionNames = extensions.map((ext) => ext.name);
        const suggestions = extensionNames.filter((name) => name.startsWith(partialArg));
        if ('--all'.startsWith(partialArg) || 'all'.startsWith(partialArg)) {
            suggestions.unshift('--all');
        }
        return suggestions;
    },
};
const exploreExtensionsCommand = {
    name: 'explore',
    description: 'Open extensions page in your browser',
    kind: CommandKind.BUILT_IN,
    action: exploreAction,
};
export const extensionsCommand = {
    name: 'extensions',
    description: 'Manage extensions',
    kind: CommandKind.BUILT_IN,
    subCommands: [
        listExtensionsCommand,
        updateExtensionsCommand,
        exploreExtensionsCommand,
    ],
    action: (context, args) => 
    // Default to list if no subcommand is provided
    listExtensionsCommand.action(context, args),
};
//# sourceMappingURL=extensionsCommand.js.map