﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalytics/KinesisAnalytics_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace KinesisAnalytics {
namespace Model {

/**
 * <p>Provides a description of an Amazon S3 data source, including the Amazon
 * Resource Name (ARN) of the S3 bucket, the ARN of the IAM role that is used to
 * access the bucket, and the name of the Amazon S3 object that contains the
 * data.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/S3Configuration">AWS
 * API Reference</a></p>
 */
class S3Configuration {
 public:
  AWS_KINESISANALYTICS_API S3Configuration() = default;
  AWS_KINESISANALYTICS_API S3Configuration(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESISANALYTICS_API S3Configuration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESISANALYTICS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>IAM ARN of the role used to access the data.</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  S3Configuration& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of the S3 bucket that contains the data.</p>
   */
  inline const Aws::String& GetBucketARN() const { return m_bucketARN; }
  inline bool BucketARNHasBeenSet() const { return m_bucketARNHasBeenSet; }
  template <typename BucketARNT = Aws::String>
  void SetBucketARN(BucketARNT&& value) {
    m_bucketARNHasBeenSet = true;
    m_bucketARN = std::forward<BucketARNT>(value);
  }
  template <typename BucketARNT = Aws::String>
  S3Configuration& WithBucketARN(BucketARNT&& value) {
    SetBucketARN(std::forward<BucketARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the object that contains the data.</p>
   */
  inline const Aws::String& GetFileKey() const { return m_fileKey; }
  inline bool FileKeyHasBeenSet() const { return m_fileKeyHasBeenSet; }
  template <typename FileKeyT = Aws::String>
  void SetFileKey(FileKeyT&& value) {
    m_fileKeyHasBeenSet = true;
    m_fileKey = std::forward<FileKeyT>(value);
  }
  template <typename FileKeyT = Aws::String>
  S3Configuration& WithFileKey(FileKeyT&& value) {
    SetFileKey(std::forward<FileKeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleARN;
  bool m_roleARNHasBeenSet = false;

  Aws::String m_bucketARN;
  bool m_bucketARNHasBeenSet = false;

  Aws::String m_fileKey;
  bool m_fileKeyHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalytics
}  // namespace Aws
