/**
 * @file ipv6_address_no_zone.c
 * @author Michal Vasko <mvasko@cesnet.cz>
 * @brief ietf-inet-types ipv6-address-no-zone type plugin.
 *
 * Copyright (c) 2019 - 2025 CESNET, z.s.p.o.
 *
 * This source code is licensed under BSD 3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://opensource.org/licenses/BSD-3-Clause
 */

#define _GNU_SOURCE /* strndup */

#include "plugins_internal.h"
#include "plugins_types.h"

#ifdef _WIN32
# include <winsock2.h>
# include <ws2tcpip.h>
#else
#  include <arpa/inet.h>
#  if defined (__FreeBSD__) || defined (__NetBSD__) || defined (__OpenBSD__)
#    include <netinet/in.h>
#    include <sys/socket.h>
#  endif
#endif
#include <assert.h>
#include <errno.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "libyang.h"

#include "compat.h"
#include "ly_common.h"

/**
 * @page howtoDataLYB LYB Binary Format
 * @subsection howtoDataLYBTypesIPv6AddressNoZone ipv6-address-no-zone (ietf-inet-types)
 *
 * | Size (b) | Mandatory | Type | Meaning |
 * | :------  | :-------: | :--: | :-----: |
 * | 128       | yes       | `struct in6_addr *` | IPv6 address in network-byte order |
 */

static void lyplg_type_free_ipv6_address_no_zone(const struct ly_ctx *ctx, struct lyd_value *value);

/**
 * @brief Convert IP address to network-byte order.
 *
 * @param[in] value Value to convert.
 * @param[in] value_len Length of @p value.
 * @param[in] options Type store callback options.
 * @param[in,out] addr Allocated value for the address.
 * @param[out] err Error information on error.
 * @return LY_ERR value.
 */
static LY_ERR
ipv6addressnozone_str2ip(const char *value, uint32_t value_len, uint32_t options, struct in6_addr *addr, struct ly_err_item **err)
{
    LY_ERR ret = LY_SUCCESS;
    const char *addr_str;
    char *addr_dyn = NULL;

    /* get the IP terminated with zero */
    if (options & LYPLG_TYPE_STORE_DYNAMIC) {
        /* we can use the value directly */
        addr_str = value;
    } else {
        addr_dyn = strndup(value, value_len);
        addr_str = addr_dyn;
    }

    /* store the IPv6 address in network-byte order */
    if (!inet_pton(AF_INET6, addr_str, addr)) {
        ret = ly_err_new(err, LY_EVALID, LYVE_DATA, NULL, NULL, "Failed to store IPv6 address \"%s\".", addr_str);
        goto cleanup;
    }

cleanup:
    free(addr_dyn);
    return ret;
}

static void
lyplg_type_lyb_size_ipv6_address_no_zone(const struct lysc_type *UNUSED(type), enum lyplg_lyb_size_type *size_type,
        uint32_t *fixed_size_bits)
{
    *size_type = LYPLG_LYB_SIZE_FIXED_BITS;
    *fixed_size_bits = 128;
}

/**
 * @brief Implementation of ::lyplg_type_store_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static LY_ERR
lyplg_type_store_ipv6_address_no_zone(const struct ly_ctx *ctx, const struct lysc_type *type, const void *value,
        uint32_t value_size_bits, uint32_t options, LY_VALUE_FORMAT format, void *UNUSED(prefix_data), uint32_t hints,
        const struct lysc_node *UNUSED(ctx_node), struct lyd_value *storage, struct lys_glob_unres *UNUSED(unres),
        struct ly_err_item **err)
{
    LY_ERR ret = LY_SUCCESS;
    struct lyd_value_ipv6_address_no_zone *val;
    uint32_t value_size;

    /* init storage */
    memset(storage, 0, sizeof *storage);
    storage->realtype = type;

    /* check value length */
    ret = lyplg_type_check_value_size("ipv6-address-no-zone", format, value_size_bits, LYPLG_LYB_SIZE_FIXED_BITS, 128,
            &value_size, err);
    LY_CHECK_GOTO(ret, cleanup);

    if (format == LY_VALUE_LYB) {
        if ((options & LYPLG_TYPE_STORE_DYNAMIC) && LYPLG_TYPE_VAL_IS_DYN(val)) {
            /* use the value directly */
            storage->dyn_mem = (void *)value;
            options &= ~LYPLG_TYPE_STORE_DYNAMIC;
        } else {
            /* allocate value */
            LYPLG_TYPE_VAL_INLINE_PREPARE(storage, val);
            LY_CHECK_ERR_GOTO(!val, ret = LY_EMEM, cleanup);

            /* store IP address */
            memcpy(&val->addr, value, sizeof val->addr);
        }

        /* success */
        goto cleanup;
    }

    /* allocate value */
    LYPLG_TYPE_VAL_INLINE_PREPARE(storage, val);
    LY_CHECK_ERR_GOTO(!val, ret = LY_EMEM, cleanup);

    /* check hints */
    ret = lyplg_type_check_hints(hints, value, value_size, type->basetype, NULL, err);
    LY_CHECK_GOTO(ret, cleanup);

    /* get the network-byte order address, validates the value */
    ret = ipv6addressnozone_str2ip(value, value_size, options, &val->addr, err);
    LY_CHECK_GOTO(ret, cleanup);

    if (format == LY_VALUE_CANON) {
        /* store canonical value */
        if (options & LYPLG_TYPE_STORE_DYNAMIC) {
            ret = lydict_insert_zc(ctx, (char *)value, &storage->_canonical);
            options &= ~LYPLG_TYPE_STORE_DYNAMIC;
            LY_CHECK_GOTO(ret, cleanup);
        } else {
            ret = lydict_insert(ctx, value_size ? value : "", value_size, &storage->_canonical);
            LY_CHECK_GOTO(ret, cleanup);
        }
    }

cleanup:
    if (options & LYPLG_TYPE_STORE_DYNAMIC) {
        free((void *)value);
    }

    if (ret) {
        lyplg_type_free_ipv6_address_no_zone(ctx, storage);
    }
    return ret;
}

/**
 * @brief Implementation of ::lyplg_type_compare_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static LY_ERR
lyplg_type_compare_ipv6_address_no_zone(const struct ly_ctx *UNUSED(ctx), const struct lyd_value *val1,
        const struct lyd_value *val2)
{
    struct lyd_value_ipv6_address_no_zone *v1, *v2;

    LYD_VALUE_GET(val1, v1);
    LYD_VALUE_GET(val2, v2);

    if (memcmp(&v1->addr, &v2->addr, sizeof v1->addr)) {
        return LY_ENOT;
    }
    return LY_SUCCESS;
}

/**
 * @brief Implementation of ::lyplg_type_sort_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static int
lyplg_type_sort_ipv6_address_no_zone(const struct ly_ctx *UNUSED(ctx), const struct lyd_value *val1,
        const struct lyd_value *val2)
{
    struct lyd_value_ipv6_address_no_zone *v1, *v2;

    LYD_VALUE_GET(val1, v1);
    LYD_VALUE_GET(val2, v2);

    return memcmp(&v1->addr, &v2->addr, sizeof v1->addr);
}

/**
 * @brief Implementation of ::lyplg_type_print_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static const void *
lyplg_type_print_ipv6_address_no_zone(const struct ly_ctx *ctx, const struct lyd_value *value, LY_VALUE_FORMAT format,
        void *UNUSED(prefix_data), ly_bool *dynamic, uint32_t *value_size_bits)
{
    struct lyd_value_ipv6_address_no_zone *val;
    char *ret;

    LYD_VALUE_GET(value, val);

    if (format == LY_VALUE_LYB) {
        *dynamic = 0;
        if (value_size_bits) {
            *value_size_bits = sizeof val->addr * 8;
        }
        return &val->addr;
    }

    /* generate canonical value if not already */
    if (!value->_canonical) {
        /* '%' + zone */
        ret = malloc(INET6_ADDRSTRLEN);
        LY_CHECK_RET(!ret, NULL);

        /* get the address in string */
        if (!inet_ntop(AF_INET6, &val->addr, ret, INET6_ADDRSTRLEN)) {
            free(ret);
            LOGERR(ctx, LY_ESYS, "Failed to get IPv6 address in string (%s).", strerror(errno));
            return NULL;
        }

        /* store it */
        if (lydict_insert_zc(ctx, ret, (const char **)&value->_canonical)) {
            LOGMEM(ctx);
            return NULL;
        }
    }

    /* use the cached canonical value */
    if (dynamic) {
        *dynamic = 0;
    }
    if (value_size_bits) {
        *value_size_bits = strlen(value->_canonical) * 8;
    }
    return value->_canonical;
}

/**
 * @brief Implementation of ::lyplg_type_dup_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static LY_ERR
lyplg_type_dup_ipv6_address_no_zone(const struct ly_ctx *ctx, const struct lyd_value *original, struct lyd_value *dup)
{
    LY_ERR ret;
    struct lyd_value_ipv6_address_no_zone *orig_val, *dup_val;

    memset(dup, 0, sizeof *dup);

    ret = lydict_insert(ctx, original->_canonical, 0, &dup->_canonical);
    LY_CHECK_GOTO(ret, error);

    LYPLG_TYPE_VAL_INLINE_PREPARE(dup, dup_val);
    LY_CHECK_ERR_GOTO(!dup_val, ret = LY_EMEM, error);

    LYD_VALUE_GET(original, orig_val);
    memcpy(&dup_val->addr, &orig_val->addr, sizeof orig_val->addr);

    dup->realtype = original->realtype;
    return LY_SUCCESS;

error:
    lyplg_type_free_ipv6_address_no_zone(ctx, dup);
    return ret;
}

/**
 * @brief Implementation of ::lyplg_type_free_clb for the ipv6-address-no-zone ietf-inet-types type.
 */
static void
lyplg_type_free_ipv6_address_no_zone(const struct ly_ctx *ctx, struct lyd_value *value)
{
    struct lyd_value_ipv6_address_no_zone *val;

    lydict_remove(ctx, value->_canonical);
    value->_canonical = NULL;
    LYD_VALUE_GET(value, val);
    LYPLG_TYPE_VAL_INLINE_DESTROY(val);
}

/**
 * @brief Plugin information for ipv6-address-no-zone type implementation.
 *
 * Note that external plugins are supposed to use:
 *
 *   LYPLG_TYPES = {
 */
const struct lyplg_type_record plugins_ipv6_address_no_zone[] = {
    {
        .module = "ietf-inet-types",
        .revision = "2025-12-22",
        .name = "ipv6-address-no-zone",

        .plugin.id = "ly2 ipv6-address-no-zone",
        .plugin.lyb_size = lyplg_type_lyb_size_ipv6_address_no_zone,
        .plugin.store = lyplg_type_store_ipv6_address_no_zone,
        .plugin.validate_value = NULL,
        .plugin.validate_tree = NULL,
        .plugin.compare = lyplg_type_compare_ipv6_address_no_zone,
        .plugin.sort = lyplg_type_sort_ipv6_address_no_zone,
        .plugin.print = lyplg_type_print_ipv6_address_no_zone,
        .plugin.duplicate = lyplg_type_dup_ipv6_address_no_zone,
        .plugin.free = lyplg_type_free_ipv6_address_no_zone,
    },
    {0}
};
